!(function () {
  /**
   * Initialize app
   *
   * @param {Client} client Freshservice client
   */
  let debug = function () {
    // empty
  };
  function init(client) {
    let domain = null;
    let currentPage = 5;
    const params = {
      agentId: null,
      pages: currentPage,
    };

    getDomain(client)
      .then(function (d) {
        debug("Using domain:", d);
        domain = d;
        return getAgentId(client, domain);
      })
      .then(function (agentId) {
        debug("Using agentId:", agentId);
        // Get tickets
        params.agentId = agentId;
        return batchGetTicketForAgent(client, domain, params);
      })
      .then(function (tickets) {
        debug("Found tickets from batch get:", tickets);
        // Filter tickets with score and within two weeks
        const ticketWithScores = tickets.filter(hasScores);
        debug("Found tickets with scores:", ticketWithScores);
        const lastTwoWeeksTickets = ticketWithScores.filter(isWithinTwoWeeks());
        debug("Found tickets within the two last weeks:", lastTwoWeeksTickets);

        // Calculate scores
        const scores = calculateScores(lastTwoWeeksTickets.map(getScores));
        debug("Scores calculated:", scores);

        // Render stuff
        renderScores(scores);
        renderScoreDistribution(
          scores.scoreDistribution,
          lastTwoWeeksTickets.length
        );

        initLoadMoreListener(
          ticketWithScores,
          function () {
            debug("Fetching more tickets");
            return getTicketsForAgent(
              client,
              domain,
              Object.assign({}, params, { page: currentPage++ })
            );
          },
          function (theseTickets) {
            theseTickets.forEach(function (ticket) {
              renderResponse(ticket, domain);
            });
          }
        );
      })
      .catch(function (e) {
        // Propagate error
        debug("Error while rendering app", e);
        throw e;
      });
  }

  // Event handlers
  /**
   * Initialize load more button handler and infite loading of tickets
   *
   * @param {string} domain Domain of current context, e.g happysignals.freshservice.com
   * @param {Object[]} initialTickets List of tickets with scores to show
   * @param {Object} getMoreTickets Callback for requesting more tickets
   * @param {Object} renderMoreResponses Callback for rendering tickets to DOM
   */
  function initLoadMoreListener(
    initialTickets,
    getMoreTickets,
    renderMoreResponses
  ) {
    debug("Init load more event listeners");
    let showAmount = 9;
    let ticketStorage = initialTickets;

    renderMoreResponses(ticketStorage.slice(0, showAmount));

    $("#loadMore").on("click", function () {
      debug("Load more clicked");
      const from = showAmount;
      showAmount += 9;
      if (ticketStorage.length < showAmount) {
        renderButtonIsLoading(true);
        getMoreTickets()
          .then(function (moreTickets) {
            debug("More tickets found:", moreTickets);
            const moreTicketsWithScores =
              moreTickets && moreTickets.length
                ? moreTickets.filter(hasScores)
                : [];
            debug("More tickets with scores found:", moreTicketsWithScores);
            renderButtonIsLoading(false);
            ticketStorage = ticketStorage.concat(moreTicketsWithScores);
          })
          .catch(function (e) {
            // Propagate error
            debug("Error while loading more tickets:", e);
            throw e;
          });
      }
      renderMoreResponses(ticketStorage.slice(from, showAmount));
    });
  }

  // Calculations
  /**
   * Calculate NPS, lost time and score distribution from array of tickets
   *
   * @param {Object[]} tickets Array of tickets with scores
   * @returns {Object}
   */
  function calculateScores(tickets) {
    debug("calculating scores for tickets", tickets);
    const stats = tickets.reduce(statsReducer, {
      positives: 0,
      negatives: 0,
      neutrals: 0,
      totalHappiness: 0,
      totalLostTime: 0,
      lostTimeTotal: 0,
      scoreDistribution: new Array(11).fill(0),
    });

    return {
      happiness:
        stats.totalHappiness > 0
          ? Math.round(
              ((stats.positives - stats.negatives) / stats.totalHappiness) * 100
            )
          : 0,
      lostTime:
        stats.totalLostTime > 0
          ? Math.round(stats.lostTimeTotal / stats.totalLostTime)
          : 0,
      scoreDistribution: stats.scoreDistribution,
    };
  }

  /**
   * Aggregate function for merging all scores to a single stat
   *
   * @param {Object} stats Aggregate for stats
   * @param {Object} score Single ticket score
   * @returns {Object}
   */
  function statsReducer(stats, score) {
    // Score distribution
    if (score.happiness) {
      stats.totalHappiness += 1;
      stats.scoreDistribution[score.happiness] += 1;
      // Nps base
      if (score.happiness < 7) {
        stats.negatives += 1;
      } else if (score.happiness > 8) {
        stats.positives += 1;
      } else {
        stats.neutrals += 1;
      }
    }

    // Lost time
    if (score.lostTime) {
      stats.lostTimeTotal += score.lostTime;
      stats.totalLostTime += 1;
    }

    return stats;
  }

  // Ticket information getters
  /**
   * Get scores for single ticket
   *
   * @param {Object} ticket Ticket
   * @returns {Object}
   */
  function getScores(ticket) {
    const happyFields = Object.keys(ticket.custom_fields || {});
    const happinessKey =
      happyFields.find(function (key) {
        return key.includes("happysignals_score");
      }) || "__not-found__";
    const lostTimeKey =
      happyFields.find(function (key) {
        return key.includes("happysignals_lost_time");
      }) || "__not-found__";
    return {
      happiness: ticket.custom_fields[happinessKey],
      lostTime: ticket.custom_fields[lostTimeKey],
    };
  }

  /**
   * Get feedback for single ticket
   *
   * @param {Object} ticket Ticket
   * @returns {string | undefined}
   */
  function getFeedback(ticket) {
    const happyFields = Object.keys(ticket.custom_fields || {});
    const feedbackKey =
      happyFields.find(function (key) {
        return key.includes("happysignals_comments");
      }) || "__not-found__";
    return ticket.custom_fields[feedbackKey];
  }

  /**
   * Get factors for single ticket
   *
   * @param {Object} ticket Ticket
   * @returns {string|undefined}
   */
  function getFactors(ticket) {
    const happyFields = Object.keys(ticket.custom_fields || {});
    const factorKey =
      happyFields.find(function (key) {
        return key.includes("happysignals_factors");
      }) || "__not-found__";
    return ticket.custom_fields[factorKey];
  }

  // Filters
  /**
   * Filter for tickets with scores
   *
   * @param {Object} ticket Ticket
   * @returns {boolean}
   */
  function hasScores(ticket) {
    const scores = getScores(ticket);
    return !!scores.happiness;
  }

  /**
   * Filter for tickets belonging to agent
   *
   * @param {number|null} id Agent id
   * @returns {Object}
   */
  function forAgent(id) {
    return function (ticket) {
      if (!id) return true;
      debug(
        "responder_id",
        ticket.responder_id,
        ticket.responder_id === id ? "equals" : "is not equal to",
        id,
        "for ticket",
        ticket.id
      );
      return ticket.responder_id && ticket.responder_id === id;
    };
  }

  /**
   * Filter for tickets for last two weeks
   *
   * @param {Object} ticket Ticket
   * @returns {Object}
   */
  function isWithinTwoWeeks() {
    const now = new Date();
    const from = new Date();
    from.setDate(now.getDate() - 14);
    return function (ticket) {
      const targetDate =
        ticket && ticket.stats && ticket.stats.resolved_at
          ? ticket.stats.resolved_at
          : null;
      return new Date(targetDate) >= from;
    };
  }

  /**
   * Get mood from happiness score
   *
   * @param {number} value Happiness score
   * @returns {string}
   */
  function getMood(value) {
    if (value < 7) {
      return "negative";
    }
    if (value > 8) {
      return "positive";
    }
    return "neutral";
  }

  // Formatting
  /**
   * Format happiness score
   *
   * @param {number} value Happiness score
   * @returns {string}
   */
  function formatNps(value) {
    if (Number.isNaN(Number(value))) {
      return "-";
    }
    return value >= 0
      ? `+${value > 100 ? 100 : value}`
      : `-${value < -100 ? 100 : Math.abs(value)}`;
  }

  /**
   * Format lost time
   *
   * @param {number} minutes Lost time in minutes
   * @returns {string}
   */
  function formatLostTime(minutes) {
    if (Number.isNaN(Number(minutes))) {
      return "-";
    }
    const days = Math.floor(minutes / (60 * 8));
    const hours = Math.floor((minutes - days * 60 * 8) / 60);
    const min = Math.round(minutes - days * 60 * 8 - hours * 60);
    return [
      { suffix: "h", value: hours },
      { suffix: "d", value: days },
    ].reduce(function (acc, part) {
      if (part.value > 0) {
        return `${part.value}${part.suffix} ${acc}`;
      }
      return acc;
    }, `${min}m`);
  }

  // Renderers
  /**
   * Render happiness and lost time scores
   *
   * @param {Object} scores Scores object
   */
  function renderScores(scores) {
    debug("Rendering scores");
    const happiness = formatNps(scores.happiness);
    const lostTime = formatLostTime(scores.lostTime);

    $("#happiness").html(happiness);
    $("#lostTime").html(lostTime);
  }

  /**
   * Render load more button loading state
   *
   * @param {boolean} disabled Should button be in loading state
   */
  function renderButtonIsLoading(isLoading) {
    debug("Rendering button loading:", isLoading);
    $("#loadMore").attr("disabled", isLoading);
    if (isLoading) {
      $("#loadMore").text("loading...");
    } else {
      $("#loadMore").text("Load more");
    }
  }

  /**
   * Render score distribution graph
   *
   * @param {number[]} scoreDistribution Score distribution array
   * @param {number} totalTickets Total amount of tickets with scores
   */
  function renderScoreDistribution(scoreDistribution, totalTickets) {
    debug("Rendering score distribution");
    const max = Math.max.apply(null, scoreDistribution);
    $(".my-scores__count").text(totalTickets);
    scoreDistribution.forEach(function (count, score) {
      const column = $($("#scoreColumn").html());
      const mood = getMood(score);
      $(".graph__column-label", column).text(score);
      $(".graph__column-value", column).text(count);
      $(".graph__column-bar", column)
        .addClass(mood)
        .css("height", (count / max) * 100);
      $(".my-scores__graph").append(column);
    });
  }

  /**
   * Render single ticket response row
   *
   * @param {Object} ticket Ticket to render
   * @param {string} domain Domain of current context, e.g happysignals.freshservice.com
   */
  function renderResponse(ticket, domain) {
    // Clonse template
    const response = $($("#response").html());

    // Render values to template
    // Scores
    const scores = getScores(ticket);
    $(".response__score", response).html(
      `<span class="${getMood(scores.happiness)}">${scores.happiness}</span>`
    );
    $(".response__lost-time", response).text(formatLostTime(scores.lostTime));

    // Meta
    $(".response__ticket-number span a", response)
      .text(`${ticket.type === "Incident" ? "#INC" : ""} ${ticket.id}`)
      .attr("href", `https://${domain}/helpdesk/tickets/${ticket.id}`);

    const requester =
      ticket.requester && ticket.requester.name ? ticket.requester.name : "-";
    $(".response__requester span", response)
      .text(requester)
      .attr("title", requester);
    $(".response__subject span", response)
      .text(ticket.subject)
      .attr("title", ticket.subject);

    // Feedback and factors
    const feedback = getFeedback(ticket);
    $(".response__feedback .feedback__text", response)
      .text(feedback)
      .attr("title", feedback);
    const factors = getFactors(ticket);
    $(".response__feedback .feedback__factors", response)
      .text(factors)
      .attr("title", factors);

    // Render item to table
    $("#responses-body").append(response);
  }

  function preInit(client) {
    return getParams(client).then((params) => {
      if (params.debug) {
        debug = console.debug;
      }
      return client;
    });
  }

  function getParams(client) {
    return client.iparams.get("debug").then(
      function (data) {
        return data;
      },
      function (error) {
        console.log(error);
      }
    );
  }

  // Client
  /**
   * Get agent Id for current user
   *
   * @param {Object} client Freshservice client
   * @return {number}
   */
  function getAgentId(client) {
    debug("Fetching agentId");
    return client.data.get("loggedInUser").then(function (user) {
      return user.loggedInUser.user_id;
    });
  }

  /**
   * Get domain for current user
   *
   * @param {Object} client Freshservice client
   * @return {number}
   */
  function getDomain(client) {
    debug("Fetching domain");
    return client.data.get("domainName").then(function (data) {
      return data.domainName;
    });
  }

  /**
   * Batch get many pages of tickets
   *
   * @param {Object} client Freshservice client
   * @param {string} domain Domain of current context, e.g happysignals.freshservice.com
   * @param {Object} params Request params
   * @return {Object[]}
   */
  function batchGetTicketForAgent(client, domain, params) {
    debug("Batch fetching tickets for agent:", params);
    return Promise.all(
      new Array(params.pages).fill(null).map(function (x, i) {
        return getTicketsForAgent(
          client,
          domain,
          Object.assign({}, params, { page: i + 1 }) // start from 1
        );
      })
    ).then(function (ticketResponses) {
      return ticketResponses.reduce(function (acc, tickets) {
        return acc.concat(tickets);
      }, []);
    });
  }

  /**
   * Get one page of tickets
   *
   * @param {Object} client Freshservice client
   * @param {string} domain Domain of current context, e.g happysignals.freshservice.com
   * @param {Object} params Request params
   * @return {Object[]}
   */
  function getTicketsForAgent(client, domain, params) {
    debug("Fetching tickets", params);
    var options = {
      method: "GET",
      headers: {
        Authorization: "Basic <%= encode(iparam.apiKey) %>",
      },
    };
    return client.request
      .get(
        encodeURI(
          `https://${domain}/api/v2/tickets?include=requester&include=stats&per_page=100&page=${
            params.page || 1
          }`
        ),
        options
      )
      .then(function (data) {
        var parsed = JSON.parse(data.response).tickets.filter(
          forAgent(params.agentId)
        );
        debug("Fetch response tickets", params, parsed);
        return parsed;
      });
  }

  // Bootstrap application
  document.addEventListener("DOMContentLoaded", function () {
    app
      .initialized()
      .then(preInit)
      .then(init)
      .catch(function (e) {
        debug("Fatal error, shutting down");
        // Silently die :(
        console.error(e);
      });
  });
})();
