const sha256 = require("js-sha256");
const VERSION = require("./version");
const FreshApi = require("./api").FreshApi;
const BUTTON_TEMPLATE = require("./template");

const DEFAULT_MAPPINGS = {
  var1: "id", // ticket id
  var2: "type_name", // ticket type (incident,request)
  // "var3": caller location
  var4: "source_name", // contact type (phone, email, portal...)
  // "ap1": company
  // "ap2": reassignment count
  // "ap3": time to resolve
  // "ap4": time worked
  ap5: "group.name", // assignment group name
  ap6: "priority_name", // priority
  // "co1": number of resolved
  // "co2": made SLA
  // "co3": country
  // "co4": configuration item name
  // "co5": service name
  // "co6": agent id
  // "lang": language
  t: "type_name",
  esm: "department_name",
  cat2: "category",
  cat3: "sub_category",
  oa: "created_at",
  d: "updated_at",
  sn: "!freshservice",
  euid: "@requester.email",
  esa: "@requester.created_at"
};


const DEFAULT_STATUS_ID = "4";

// Get resolution status ID for a specific ticket type
function getResolutionStatusForType(configStr, ticketType) {
  if (!configStr || configStr.trim() === '') return null;

  // Single status ID (unescaped, e.g., "4")
  if (/^\d+$/.test(configStr)) {
    return configStr;
  }

  try {
    const config = JSON.parse(configStr);

    // Single status ID (parsed, e.g., "4" from "\"4\"")
    if (typeof config === 'string' && /^\d+$/.test(config)) {
      return config;
    }

    // JSON object with ticket types
    if (typeof config === 'object' && config !== null && ticketType in config) {
      const statusId = config[ticketType];
      if (/^\d+$/.test(statusId)) {
        return statusId;
      }
      console.info(`Invalid status ID for ticket type ${ticketType}: '${statusId}' is not a number (e.g., 4 or 6). Using default ID ${DEFAULT_STATUS_ID}.`);
      return DEFAULT_STATUS_ID;
    }

    // Invalid format or missing ticket type
    console.info(`Invalid or missing configuration for ticket type ${ticketType} in '${configStr}'. Using default ID ${DEFAULT_STATUS_ID}.`);
    return DEFAULT_STATUS_ID;
  } catch (e) {
    console.info(`Failed to parse resolutionConfigByType: '${configStr}' is invalid JSON. Using default ID ${DEFAULT_STATUS_ID}.`);
    return DEFAULT_STATUS_ID;
  }
}

// Check if ticket is in a resolved state
function isResolved(ticket, statusId) {
  if (statusId === null) return false;
  const resolvedStatusId = parseInt(statusId, 10);
  return ticket.status === resolvedStatusId;
}

// Check if ticket update is a resolution event
function isResolutionUpdate(ticketChanges, statusId, debug) {
  if (!ticketChanges.status) return false;
  if (statusId === null) return false;
  const resolvedStatusId = parseInt(statusId, 10);
  const [oldStatus, newStatus] = ticketChanges.status;
  debug && console.info(`Status changed from ${oldStatus} to ${newStatus}, resolved ID: ${resolvedStatusId}`);
  return oldStatus !== resolvedStatusId && newStatus === resolvedStatusId;
}

function isTicketResolved(event, ticket, ticketChanges, statusId, debug) {
  let isTicketInResolvedState = false;

  if (event === "onTicketCreate") {
    isTicketInResolvedState = isResolved(ticket, statusId);
    debug && console.info(`onTicketCreate: isResolved=${isTicketInResolvedState}`);
  } else if (event === "onTicketUpdate") {
    isTicketInResolvedState = isResolutionUpdate(ticketChanges, statusId, debug);
    debug && console.info(`onTicketUpdate: isResolved=${isTicketInResolvedState}`);
  }

  return isTicketInResolvedState;
}


exports = function (args) {
  const ticket = args.data.ticket;
  const client = new FreshApi(args);
  const debug = !!args.iparams.debug;
  const bypassMandatoryChecks = !!args.iparams.bypassMandatoryChecks;

  // Get resolution status ID for this ticket type
  const statusId = getResolutionStatusForType(args.iparams.resolutionConfigByType, ticket.type_name);
  debug && console.info("Version " + VERSION + " updating ticket " + ticket.id);

  const ticketChanges = ticket.changes || {};
  const changedStandardFields = Object.keys(ticketChanges).filter(
    (t) => t !== "custom_fields"
  );
  const changedCustomFields = (ticketChanges.custom_fields || []).map(
    (t) => t.name
  );
  const changedFields = changedStandardFields.concat(changedCustomFields);

  if (changedFields.length === 0 && args.event === "onTicketUpdate") {
    console.info("onTicketUpdate event with no changes listed, stopping");
    return;
  }
  if (
    changedFields.find((field) =>
      field
        .replace(/[^a-z]+/gi, "")
        .toLowerCase()
        .match(/^happysignals/)
    )
  ) {
    console.info(
      "Event contained update to HappySignals Buttons field, stopping"
    );
    return;
  }
  debug &&
    console.info(
      "The following fields were updated; " + changedFields.join(", ")
    );

  // Check for resolution event
  const isResolutionEvent = isTicketResolved(args.event, ticket, ticketChanges, statusId, debug);
  if (!isResolutionEvent && args.event === "onTicketUpdate") {
    debug && console.info("Not a resolution event, skipping pre-response");
    return;
  }

  computeParameterMappings(args.iparams.customMappings)
    .then(mapParameters)
    .then(createStub)
    .then((stub) => {
      const promises = [updateTicket(stub)];
      if (isResolutionEvent) {
        promises.push(sendPreResponse(stub));
      }
      return Promise.all(promises);
    })
    .then(
      function () {
        console.info("Successfully updated ticket", ticket.id);
      },
      function (response) {
        console.error("Unable to update ticket", ticket.id, response.message);
        (function () {
          /* Look! We're not "simply logging errors" */
        })();
      }
    );

  function computeParameterMappings(customMappings) {
    const defaultMappings = Object.keys(DEFAULT_MAPPINGS).map((key) => ({
      field: key,
      mapping: DEFAULT_MAPPINGS[key],
    }));
    const customMappingsList = (customMappings || "")
      .split(/[,;\n]/)
      .map((t) => t.replace(/^\s+|\s+$/g, ""))
      .filter((t) => t.length > 0 && t.substring(0, 1) !== "#")
      .map((t) => t.split("=").map((t) => t.replace(/^\s+|\s+$/g, "")))
      .map((pair) => ({ field: pair[0], mapping: pair[1] }));
    const mappings = defaultMappings.concat(customMappingsList);
    debug &&
      console.info(
        "Using mapping",
        mappings.map((t) => t.mapping + "->" + t.field).join(", ")
      );
    return Promise.resolve(mappings);
  }

  function mapParameters(mappings) {
    return Promise.all(
      mappings.map((pair) => {
        const key = pair.field;
        const mapping = pair.mapping;
        if (mapping[0] === "!") {
          // Constant
          return { key: key, value: mapping.substring(1) };
        } else if (mapping.substring(0, 14) === "custom_fields.") {
          // From custom field on ticket
          return {
            key: key,
            value: (ticket.custom_fields || [])
              .filter((t) => t.name == mapping.substring(14))
              .map((t) => t.value)[0],
          };
        } else if (mapping.substring(0, 6) === "group.") {
          // From group field
          if (!ticket.group_id) {
            console.log("No group assigned to ticket");
            return { key: key };
          }
          return client.getGroup(ticket.group_id).then(
            (group) => ({
              key: key,
              value: group[mapping.substring(6)],
            }),
            () => {
              console.error("Unable to find group", ticket.group_id);
              return { key: key };
            }
          );
        } else if (mapping.substring(0, 11) === "@requester.") {
          // From requester field
          return client.getRequesterOrAgent(args.data.requester.id).then(
            (requester) => {
              // Find what property to map, possibly nested
              const subkey = mapping
                .substring(11)
                .match(/^(\w+)(?:\[(\d+)\])?(?:\.(\w+))?/); // mapping
              if (subkey[1] === "email") {
                // Special handling for euid parameter
                if (!requester.primary_email) {
                  debug && console.info("No primary email available for requester");
                  return { key: key };
                }
                // Calculate euid as first 10 chars of SHA256(email)
                const euid = sha256(requester.primary_email).substring(0, 10);
                debug && console.info("Calculated euid for email:", requester.primary_email, "euid:", euid);
                return { key: key, value: euid };
              } else if (subkey[1] === "location") {
                // Location field
                if (!requester.location_id) {
                  debug &&
                    console.info(
                      "No location available on requester for field",
                      subkey[3]
                    );
                  return { key: key };
                }
                return client
                  .getLocation(requester.location_id)
                  .then((location) => {
                    return { key: key, value: location[subkey[3]] };
                  });
              } else if (subkey[1] === "department" && subkey[2]) {
                // Department field
                return client
                  .getDepartment(requester.department_ids[subkey[2]])
                  .then((department) => {
                    return { key: key, value: department[subkey[3]] };
                  });
              } else if (subkey[1] === "custom_fields" && subkey[3]) {
                // custom field on requester
                return { key: key, value: requester.custom_fields[subkey[3]] }; // subkey structure is the following ["custom_fields.field_name", "custom_fields","null", "field_name"]
              } else if (subkey[1] === "custom_field" && subkey[3]) {
                // EDF has custom fields on requester, but these are returned with non-standard notation of custom_field : {cf_field_name}
                return { key: key, value: requester.custom_field[subkey[3]] }; // subkey structure is the following ["custom_field.field_name", "custom_field","null", "field_name"]
              } else {
                // Field on requester
                return { key: key, value: requester[subkey[1]] };
              }
            },
            () => {
              console.error("Unable to find ticket requester");
              return { key: key };
            }
          );
        } else {
          // From ticket field
          return { key: key, value: ticket[mapping] };
        }
      })
    ).then((fields) => {
      // Reduce to object
      const parameters = fields.reduce((acc, field) => {
        if (field.key === 'euid' && field.value) {
          // Hash any value mapped to euid
          acc[field.key] = sha256(field.value).substring(0, 10);
          debug && console.info("Calculated euid hash for value:", field.value);
        } else {
          acc[field.key] = field.value || acc[field.key]; // Last value in wins
        }
        return acc;
      }, {});
      debug &&
        console.info(
          "Mapped parameters",
          Object.keys(parameters)
            .map((t) => t + "=" + parameters[t])
            .join(", ")
        );
      return parameters;
    });
  }

  function createStub(parameters) {
    const signature = sha256.hmac(
      args.iparams.sharedSecret,
      [parameters["var1"], parameters["esm"], parameters["t"], parameters["d"]]
        .map(function (value) {
          return value || "";
        })
        .join("")
    );
    debug && console.info("Calculated signature", signature);

    // Update parameters with signature and build link stub
    parameters["c"] = signature;
    let stub =
      args.iparams.instanceUrl +
      "/a/happy?" +
      Object.keys(parameters)
        .map(function (key) {
          return (
            encodeURIComponent(key) +
            "=" +
            encodeURIComponent(parameters[key] || "")
          );
        })
        .join("&");

    if (!stub.match(/^https?\:\/\//)) {
      stub = "https://" + stub;
    }

    debug && console.info("Link stub is", stub);
    return Promise.resolve(stub);
  }

  function updateTicket(stub) {
    return client.getCustomFieldKeys(["HappySignals Buttons"]).then(
      function (field_keys) {
        const buttons_key = field_keys["HappySignals Buttons"];
        if (!buttons_key) {
          debug &&
            console.info(
              "No happy signals buttons field found, available fields are " +
              fields.map((t) => t.ticket_field.label).join(", ")
            );
          throw new Error("No 'HappySignals Buttons' field configured");
        }

        // Assemble a custom fields object with update on our field
        debug && console.info("Storing buttons in field", buttons_key);
        const custom_fields = {};
        custom_fields[buttons_key] = BUTTON_TEMPLATE.replace(
          /\{\{([^}]+)\}\}/g,
          function (_, keyword) {
            return keyword === "bgColor"
              ? args.iparams.buttonColor || "#ef80b2"
              : //keyword==="robotCatcher" ? stub + "&ua=robot":
              keyword.substring(0, 5) === "link_"
                ? stub + "&value=" + keyword.substring(5)
                : ""; // Unknown keyword
          }
        );

        // Update ticket
        const bypassParameter = bypassMandatoryChecks
          ? "bypass_mandatory=true"
          : "";
        return client.put(
          "/api/v2/tickets/" + ticket.id + "?" + bypassParameter,
          {
            ticket: {
              custom_fields: custom_fields,
            },
          }
        );
      },
      function (err) {
        console.error("Unable to fetch custom field keys", err);
        return Promise.reject(err);
      }
    );
  }

  function sendPreResponse(stub) {
    debug && console.info("Sending value-less pre-response for ticket", ticket.id);
    const happysignalsHost = args.iparams.instanceUrl.split("//")[1];
    const path = '/' + stub.split('/').slice(3).join('/');
    debug && console.info(`Pre-response path: ${path}, host: ${happysignalsHost}`);
    return $request.invokeTemplate("makeGetRequest", {
      context: { path, happysignalsHost },
    }).then(
      function (data) {
        debug && console.info("Successfully sent value-less pre-response for ticket", ticket.id);
        return data;
      },
      function (err) {
        console.error("Failed to send value-less pre-response for ticket", ticket.id, err);
        throw err;
      }
    );
  }
};