const api = require("./api"),
  deferred = require("jquery-deferred"),
  Deferred = deferred.Deferred,
  HappyApi = api.HappyApi,
  FreshApi = api.FreshApi;

const NOTE_TEMPLATE = require("./note-template")

exports = function pollFeedback(args) {
  const happyClient = new HappyApi(args);
  const freshClient = new FreshApi(args);
  let fieldKeys = {};
  const debug = !!args.iparams.debug;

  debug && console.info("Starting to poll for new responses from HappySignals")
  return freshClient
    // Fetch the keys for our custom fields
    .getCustomFieldKeys([
      "HappySignals Score",
      "HappySignals Lost Time",
      "HappySignals Factors",
      "HappySignals Comment",
      "HappySignals Profile",
      "HappySignals Can Be Contacted"
    ])
    .then(function (res) {
      debug && console.info("Retrieved custom field keys ", res)
      fieldKeys = res;
    })
    // Fetch the last offset that we processed in a previous round
    .then(fetchLastOffset)
    .then(function (offset) {
      debug && console.info("Retrieved previous offset", offset);
      return offset;
    })
    // Start fetching an processing pages of new responses from HappySignals
    .then(
      processRemainingPages(),
      function (err) { console.error("Error while fetching responses", err) }
    )

  function processRemainingPages() {
    return function (timestamp) {
      debug && console.info("Fetching responses starting from timestamp", timestamp)
      return happyClient
        .get('timestamp__gt=' + timestamp)
        .then(function (data) {
          debug && console.info("Received", data.items.length, "response(s) in batch");

          // Break if no more responses received
          if (!data.items || !data.items.length) {
            debug && console.info("No more responses to process");
            return;
          }

          return data.items
            // Chain promises to update each ticket in batch
            .reduce(
              (prev, response) => prev.then(applyFeedback(response)),
              Promise.resolve(timestamp)
            )
            // Finally request the next page
            .then(processRemainingPages())
        }
        )
        .catch(function (err) {
          console.error("Error fetching responses, error was", err);
          throw err
        })
    }
  }

  function applyFeedback(feedback) {
    return function () {
      const customFields = {};
      if (fieldKeys["HappySignals Score"]) {
        customFields[fieldKeys["HappySignals Score"]] = feedback.value; // requires number type field to be created in FreshService
      }
      if (fieldKeys["HappySignals Lost Time"]) {
        customFields[fieldKeys["HappySignals Lost Time"]] = feedback.lostTime; // requires number type field to be created in FreshService
      }
      if (fieldKeys["HappySignals Comment"]) {
        customFields[fieldKeys["HappySignals Comment"]] = feedback.comment;
      }
      if (fieldKeys["HappySignals Profile"]) {
        customFields[fieldKeys["HappySignals Profile"]] = feedback.profile && (feedback.profile.charAt(0).toUpperCase() + feedback.profile.slice(1));
      }
      if (fieldKeys["HappySignals Factors"]) {
        customFields[fieldKeys["HappySignals Factors"]] = (feedback.factors || [])
          .map(function (item) { return item.title })
          .join("\n")
      }
      if (fieldKeys["HappySignals Can Be Contacted"]) {
        customFields[fieldKeys["HappySignals Can Be Contacted"]] = feedback.canBeContacted === null ? "" : feedback.canBeContacted === true ? "Yes" : "No"; // Freshservice does not support boolean type fields
      }

      return freshClient
        // Update ticket fields
        .put("/api/v2/tickets/" + feedback.ticketId, {
          ticket: {
            custom_fields: customFields,
          },
        })
        .then(function () {
          debug && console.info("Updated fields on ticket", feedback.ticketId);
        })
        // Add private note
        .then(function () {
          if (!args.iparams.notes) {
            debug && console.info("Ticket notes disabled, not posting note on ticket", feedback.ticketId);
            return;
          }
          return freshClient.post("/api/v2/tickets/" + feedback.ticketId + "/notes", {
            incoming: true,
            body: NOTE_TEMPLATE({ ...feedback, factors: feedback.factors.map(t => t.title) })
          }).then(function () {
            debug && console.info("Posted note on ticket", feedback.ticketId);
          })
        })

        // Store last timestamp for next invocation if successful else continue this prevents a single ticket from
        // getting the process stuck in an infinite loop
        .then(
          function () { return setLastOffset(feedback.timestamp) },
          function (response) {
            console.error("Unable to update ticket", feedback.ticketId, "error was", response.message);
            return new Deferred().resolve(feedback.timestamp)
          }
        )
    }
  }

  function fetchLastOffset() {
    return $db.get("happysignals")
      .then(function (value) {
        return value.offset
      }, function (err) {
        console.error("Unable to fetch stored offset", err);
        return new Deferred().resolve(
          args.iparams.startDate
            ? new Date(args.iparams.startDate).valueOf() * 1000
            : new Date(new Date().valueOf() - (3 * 30 * 24 * 60 * 60 * 1000)).valueOf() * 1000
        );
      })
  }

  function setLastOffset(offset) {
    return $db.set("happysignals", { offset: offset })
      .then(function () {
        debug && console.log("Stored latest offset as", offset);
        return offset;
      }, function (err) {
        console.error("Unable to store offset", err);
        return new Deferred().resolve(offset)
      })
  }
}

