const base64 = require('base-64');

function HappyApi(args) {
  this.args = args;
}

HappyApi.prototype.call = function (template, timestampQuery) {
  const args = this.args;
  const debug = args.iparams.debug;
  const happysignalsHost = args.iparams.instanceUrl.split("//")[1];
  const query = !args.iparams.additionalQuery ? timestampQuery : args.iparams.additionalQuery + "&" + timestampQuery;

  debug && console.info("Initiating HappySignals API call with query:", query);

  return new Promise(function (resolve, reject) {
    $request.invokeTemplate(template, {
      context: {
        happysignalsHost: happysignalsHost,
        queryString: query
      }
    }).then(function (response) {
      if (response.status != 200)
        return reject(new Error("Expected status 200, received " + response.status + " with response: " + JSON.stringify(response, null, 4)));
      return resolve(JSON.parse(response.response))
    }, function (response) {
      return reject(new Error("Expected status 200, received " + response.status + " with response: " + JSON.stringify(response, null, 4)));
    })
  });
}

HappyApi.prototype.get = function (endpoint, json, headers) {
  return this.call("getHappySignalsResponses", endpoint, json, headers);
}

function FreshApi(args) {
  this.args = args;
  this.debug = args.iparams.debug;
  this.requestersCache = {}
  this.locationsCache = {}
  this.departmentsCache = {}
  this.groupsCache = {}
  this.templates = {
    "post": "freshServiceApiPost",
    "get": "freshServiceApiGet",
    "put": "freshServiceApiPut"
  }
}

FreshApi.prototype.call = function (method, endpoint, json, headers) {
  const args = this.args;
  const debug = this.debug;
  const template = this.templates[method];

  return new Promise(function (resolve, reject) {
    $request.invokeTemplate(template, {
      context: {
        path: endpoint,
      },
      body: JSON.stringify(json)
    }).then(function (response) {
      if (response.status < 200 || response.status >= 300) {
        return reject(new Error("Expected status 2xx, received " + response.status));
      }
      if (response.response === '{"require_login":true}') {
        return reject(new Error("Unauthorized"));
      }
      return resolve(response.response)
    },

      function (response) {
        if (response.status !== 404) {
          debug && console.info("Failed API call", args.domain, method, endpoint, "\n", {
            headers: headers, json: json
          });
          console.error("API error", response.status, "response:\n", response.response);
        }
        console.error("API error", response.status, "response:\n", JSON.stringify(response, null, 4));
        return reject(new Error("Expected status 2xx, received " + response.status + "  for url " + args.domain + endpoint));
      }
    )
  })
}
FreshApi.prototype.post = function (endpoint, json, headers) {
  return this.call("post", endpoint, json, headers)
}
FreshApi.prototype.get = function (endpoint, json, headers) {
  return this.call("get", endpoint, json, headers)
}
FreshApi.prototype.put = function (endpoint, json, headers) {
  this.debug && console.info("Updating ticket", endpoint, json);
  return this.call("put", endpoint, json, headers)
}
FreshApi.prototype.getCustomFieldKeys = function (args) {
  const find_fields = {};
  args.forEach(function (t) {
    find_fields[t.replace(/[^a-z]+/gi, "").toLowerCase()] = t;
  })
  return this.get("/api/v2/ticket_form_fields").then(function (response) {
    const found_fields = {}
    JSON.parse(response).ticket_fields.forEach(function (field) {
      const canonised_title = field.label.replace(/[^a-z]+/gi, "").toLowerCase();
      if (find_fields[canonised_title]) {
        found_fields[find_fields[canonised_title]] = field.name;
      }
    });
    return found_fields
  })
}
FreshApi.prototype.getRequesterOrAgent = function (id) {
  if (!this.requestersCache[id]) {
    this.requestersCache[id] = this.get(
      "/api/v2/requesters/" + id
    ).then(response => {
      this.debug && console.info("Retrieved requester", id);
      return JSON.parse(response).requester
    }, () => {
      // No requester, lets find an agent then
      return this.get(
        "/api/v2/agents/" + id
      ).then(response => {
        this.debug && console.info("Retrieved agent", id, response);
        return JSON.parse(response).agent
      })
    });
  }
  return this.requestersCache[id];
}
FreshApi.prototype.getLocation = function (id) {
  if (!this.locationsCache[id]) {
    this.locationsCache[id] = this.get(
      "/api/v2/locations/" + id
    ).then(response => {
      this.debug && console.info("Retrieved location", id);
      return JSON.parse(response).location
    });
  }
  return this.locationsCache[id];
}
FreshApi.prototype.getDepartment = function (id) {
  if (!this.departmentsCache[id]) {
    this.departmentsCache[id] = this.get(
      "/api/v2/departments/" + id
    ).then(response => {
      this.debug && console.info("Retrieved department", id);
      return JSON.parse(response).department
    });
  }
  return this.departmentsCache[id];
}
FreshApi.prototype.getGroup = function (id) {
  if (!this.groupsCache[id]) {
    this.groupsCache[id] = this.get(
      "/api/v2/groups/" + id
    ).then(response => {
      this.debug && console.info("Retrieved group", id, response);
      return JSON.parse(response).group
    });
  }
  return this.groupsCache[id];
}

exports = { FreshApi: FreshApi, HappyApi: HappyApi }